import { PanelDetailModel } from './panel-detail';

/**
 * 直接内容模型
 *
 * @export
 * @class PanelRawitemModel
 * @extends {PanelDetailModel}
 */
export class PanelRawitemModel extends PanelDetailModel {

    /**
     * 导航数据（用于导航区占位）
     *
     * @private
     * @type {*}
     * @memberof PanelRawitemModel
     */
    public navData: any = {};

    /**
     * 视图类型
     *
     * @private
     * @type {string}
     * @memberof PanelRawitemModel
     */
    private readonly viewType: string;

    /**
     * 绘制模式
     *
     * @private
     * @type {('TEXT' | 'HEADING1' | 'HEADING2' | 'HEADING3' | 'HEADING4' | 'HEADING5' | 'HEADING6' | 'PARAGRAPH')}
     * @memberof PanelRawitemModel
     */
    public renderMode: 'TEXT' | 'HEADING1' | 'HEADING2' | 'HEADING3' | 'HEADING4' | 'HEADING5' | 'HEADING6' | 'PARAGRAPH' = 'TEXT';

    /**
     * 内容类型
     *
     * @private
     * @type {('RAW' | 'HTML' | 'IMAGE' | 'MARKDOWN')}
     * @memberof PanelRawitemModel
     */
    public contentType: 'RAW' | 'HTML' | 'IMAGE' | 'MARKDOWN' = 'RAW';

    /**
     * 内容样式
     *
     * @type {string}
     * @memberof PanelRawitemModel
     */
    public contentStyle!: string;

    /**
     * 直接内容
     *
     * @type {string}
     * @memberof PanelRawitemModel
     */
    public rawContent!: string;

    /**
     * html内容
     *
     * @type {string}
     * @memberof PanelRawitemModel
     */
    public htmlContent!: string;

    /**
     * 换行模式{WRAP：换行、 NOWRAP：不换行 }
     *
     * @type {(string | 'WRAP' | 'NOWRAP')}
     * @memberof PanelRawitemModel
     */
    public wrapMode: string | 'WRAP' | 'NOWRAP' = 'WRAP';

    /**
     * 文本垂直对齐模式{TOP：上对齐、 MIDDLE：居中、 BOTTOM：下对齐 }
     *
     * @type {(string | 'TOP' | 'MIDDLE' | 'BOTTOM')}
     * @memberof PanelRawitemModel
     */
    public vAlign: string | 'TOP' | 'MIDDLE' | 'BOTTOM' = 'MIDDLE';

    /**
     * 文本水平对齐模式{LEFT：左对齐、 CENTER：居中、 RIGHT：右对齐、 JUSTIFY：两端对齐 }
     *
     * @type {(string | 'LEFT' | 'CENTER' | 'RIGHT' | 'JUSTIFY')}
     * @memberof PanelRawitemModel
     */
    public hAlign: string | 'LEFT' | 'CENTER' | 'RIGHT' | 'JUSTIFY' = 'LEFT';

    /**
     * Creates an instance of PanelRawitemModel.
     * @param {*} [opts={}]
     * @memberof PanelRawitemModel
     */
    constructor(opts: any = {}) {
        super(opts);
        this.viewType = opts.viewType;
        this.renderMode = opts.renderMode ? opts.renderMode : 'TEXT';
        this.contentType = opts.contentType ? opts.contentType : 'RAW';
        this.contentStyle = opts.contentStyle;
        this.rawContent = opts.rawContent;
        this.htmlContent = opts.htmlContent;
        this.wrapMode = opts.wrapMode ? opts.wrapMode : 'WRAP';
        this.vAlign = opts.vAlign ? opts.vAlign : 'MIDDLE';
        this.hAlign = opts.hAlign ? opts.hAlign : 'LEFT';
    }

    /**
     * 设置导航数据
     *
     * @param {*} value
     * @memberof PanelRawitemModel
     */
    setNavData(value: any) {
        this.navData = value;
    }

    /**
     * 获取导航数据
     *
     * @return {*} 
     * @memberof PanelRawitemModel
     */
    getNavData() {
        return this.navData;
    }

    /**
     * 获取动态导航模式（DYNAMICCOMP：动态组件 ROUTEVIEW：路由出口）
     *
     * @readonly
     * @type {('DYNAMICCOMP' | 'ROUTEVIEW')}
     * @memberof PanelRawitemModel
     */
    get dynaNavMode(): 'DYNAMICCOMP' | 'ROUTEVIEW' {
        return this.viewType === 'APPINDEXVIEW' ? 'ROUTEVIEW' : 'DYNAMICCOMP';
    }

    /**
     * 是否启用动态缓存
     *
     * @readonly
     * @type {boolean}
     * @memberof PanelRawitemModel
     */
    get enableCache(): boolean {
        if (this.viewType === 'APPINDEXVIEW' && this.panel && this.panel.layoutModelDetails) {
            const navPos = Object.values(this.panel.layoutModelDetails).find((item: any) => {
                return item.predefinedType === 'NAV_TABS';
            })
        }
        return false;
    }

    /**
     * 获取元素样式（直接内容元素，包含内容盒子 大小/边距/内容 的样式）
     *
     * @return {*} 
     * @memberof PanelRawitemModel
     */
    public getElementStyle() {
        const elementStyle = {};
        Object.assign(elementStyle, this.getBoxSizeStyle());
        Object.assign(elementStyle, this.getBoxSpacingStyle());
        Object.assign(elementStyle, this.getBoxContentStyle());
        Object.assign(elementStyle, this.getBoxSelfAlignStyle());
        return elementStyle;
    }

    /**
     * 获取文本内容布局样式
     *
     * @protected
     * @memberof PanelRawitemModel
     */
    protected getBoxContentStyle() {
        const contentStyle = {};
        // 文本换行模式
        if (this.wrapMode) {
            switch (this.wrapMode) {
                case 'NOWRAP':
                    Object.assign(contentStyle, { 'white-space': 'nowrap', 'text-overflow': 'ellipsis', 'overflow': 'hidden' });
                    break;
                default:
                    break;
            }
        }
        // 文本水平对齐
        if (this.hAlign) {
            switch (this.hAlign) {
                case 'LEFT':
                    Object.assign(contentStyle, { 'text-align': 'left' });
                    break;
                case 'CENTER':
                    Object.assign(contentStyle, { 'text-align': 'center' });
                    break;
                case 'RIGHT':
                    Object.assign(contentStyle, { 'text-align': 'right' });
                    break;
                case 'JUSTIFY':
                    Object.assign(contentStyle, { 'text-align': 'justify' });
                    break;
                default:
                    break;
            }
        }
        // 文本垂直对齐
        if (this.vAlign) {
            switch (this.vAlign) {
                case 'TOP':
                    Object.assign(contentStyle, { 'vertical-align': 'top' });
                    break;
                case 'MIDDLE':
                    Object.assign(contentStyle, { 'vertical-align': 'middle' });
                    break;
                case 'BOTTOM':
                    Object.assign(contentStyle, { 'vertical-align': 'bottom' });
                    break;
                default:
                    break;
            }
        }
        return contentStyle;
    }
}