import { PanelDetailModel } from './panel-detail';

/**
 * 面板容器模型
 *
 * @export
 * @class PanelContainerModel
 * @extends {PanelDetailModel}
 */
export class PanelContainerModel extends PanelDetailModel {

    /**
     * 子项
     *
     * @type {string[]}
     * @memberof PanelContainerModel
     */
    public details: string[] = [];

    /**
     * 标题关闭模式
     *
     * @type {number}
     * @memberof PanelContainerModel
     */
    public titleBarCloseMode: number = 0;

    /**
     * 背景图片
     *
     * @type {*}
     * @memberof PanelContainerModel
     */
    public sysImage: any = {};

    /**
     * 数据区域类型
     * @description 值模式 [数据面板模式] {NONE：无、 LOGINFORM：登录表单、 SINGLEDATA：单项数据、 MULTIDATA：多项数据、 INHERIT：继承、 USER：用户自定义 }
     * @type {( string | 'NONE' | 'LOGINFORM' | 'SINGLEDATA' | 'MULTIDATA' | 'INHERIT' | 'USER')} 
     * @memberof PanelContainerModel
     */
    public dataRegionType: string | 'NONE' | 'LOGINFORM' | 'SINGLEDATA' | 'MULTIDATA' | 'INHERIT' | 'USER' | '' = '';

    /**
     * 数据源类型
     * @description 值模式 [数据面板源（全部）] {DEACTION：实体行为、 DEDATASET：实体集合、 DELOGIC：实体逻辑、 APPGLOBALPARAM：绑定应用全局变量、 TOPVIEWSESSIONPARAM：绑定顶级视图会话共享变量、 VIEWSESSIONPARAM：绑定当前视图会话共享变量 }
     * @type {( string | 'DEACTION' | 'DEDATASET' | 'DELOGIC' | 'APPGLOBALPARAM' | 'TOPVIEWSESSIONPARAM' | 'VIEWSESSIONPARAM')} 
     * @memberof PanelContainerModel
     */
    public dataSourceType: string | 'DEACTION' | 'DEDATASET' | 'DELOGIC' | 'APPGLOBALPARAM' | 'TOPVIEWSESSIONPARAM' | 'VIEWSESSIONPARAM' | '' = '';

    /**
     * 应用实体codeName
     *
     * @type {(string | undefined)}
     * @memberof PanelContainerModel
     */
    public appDataEntityCodeName: string | undefined = undefined;

    /**
     * 应用实体方法CodeName
     *
     * @type {(string | undefined)}
     * @memberof PanelContainerModel
     */
    public appDEMethodCodeName: string | undefined = undefined;

    /**
     * Creates an instance of PanelContainerModel.
     * @param {*} [opts={}]
     * @memberof PanelContainerModel
     */
    public constructor(opts: any = {}) {
        super(opts);
        this.details = opts.details;
        this.titleBarCloseMode = opts.titleBarCloseMode;
        this.sysImage = opts.sysImage;
        this.appDataEntityCodeName = opts.appDataEntityCodeName;
        this.appDEMethodCodeName = opts.appDEMethodCodeName;
        this.dataRegionType = opts.dataRegionType;
        this.dataSourceType = opts.dataSourceType;
    }

    /**
     * 获取容器元素样式(仅限容器元素)
     *
     * @return {*} 
     * @memberof PanelContainerModel
     */
    public getElementStyle() {
        const containerStyle = {};
        Object.assign(containerStyle, this.getFlexStyle());
        //边缘布局容器盒子大小由布局组件内部计算
        if (this.layout !== "BORDER") {
            Object.assign(containerStyle, this.getBoxSizeStyle());
        }
        Object.assign(containerStyle, this.getBoxSpacingStyle());
        Object.assign(containerStyle, this.getSpecificStyle());
        return containerStyle;
    }

    /**
     * 获取容器特有样式（主要包含背景图片）
     *
     * @protected
     * @memberof PanelContainerModel
     */
    protected getSpecificStyle() {
        const boxStyle = {};
        if (Object.is(this.itemStyle, 'STYLE2') && this.sysImage && (this.sysImage.imagePath || this.sysImage.rawContent)) {
            Object.assign(boxStyle, { background: `url('${this.sysImage.imagePath || this.sysImage.rawContent}') no-repeat 0px 0px`, 'background-size': '100% 100%' });
            if (!this.layoutWidth) {
                Object.assign(boxStyle, { width: '100%' });
            }
            if (!this.layoutHeight) {
                Object.assign(boxStyle, { height: '100%' });
            }
        }
        return boxStyle;
    }

    /**
     * 加载
     *
     * @param {*} context
     * @param {*} viewParams
     * @memberof PanelContainerModel
     */
    public async load(context: any, viewParams: any) {
        const nullData = {};
        switch (this.dataRegionType) {
            case 'LOGINFORM':
                this.setData(nullData);
                break;
            case 'INHERIT':
                if (!this.parentName) {
                    this.setData(nullData);
                } else {
                    const parentItem = this.panel.layoutModelDetails[this.parentName];
                    if (this.parentName) {
                        if (parentItem.dataRegionType === 'MULTIDATA' && parentItem['data'] && parentItem['data'].length > 0) {
                            this.setData(parentItem.getData()[this.getIndex()]);
                        } else {
                            this.setData(parentItem.getData());
                        }
                    }
                }
                break;
            case 'SINGLEDATA':
            case 'MULTIDATA':
                this.setData(await this.loadData(context, viewParams));
                break;
            default:
                console.warn(`${this.dataRegionType}数据域未实现`);
                break;
        }
    }

    /**
     * 加载数据
     *
     * @param {*} context
     * @param {*} viewParams
     * @return {*} 
     * @memberof PanelContainerModel
     */
    public async loadData(context: any, viewParams: any) {
        let data = {};
        if (this.dataSourceType === 'DEACTION' || this.dataSourceType === 'DEDATASET') {
            try {
                if (this.appDataEntityCodeName && this.appDEMethodCodeName) {
                    const service = await window.entityServiceRegister.getService(this.appDataEntityCodeName);
                    if (service) {
                        if (service[this.appDEMethodCodeName] && service[this.appDEMethodCodeName] instanceof Function) {
                            const response = await service[this.appDEMethodCodeName](context, viewParams);
                            if (response && response.status === 200) {
                                data = response.data;
                            }
                        }
                    }
                }
            } catch (error) {
                console.error(`${this.dataSourceType}数据源请求数据异常`);
            }
        } else {
            console.warn(`${this.dataSourceType}数据源类型未实现`);
        }
        return data;
    }
}