package cn.ibizlab.demoapi.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.demoapi.dto.*;
import cn.ibizlab.demoapi.mapping.*;
import cn.ibizlab.core.sample.domain.IBIZCPU;
import cn.ibizlab.core.sample.service.IIBIZCPUService;
import cn.ibizlab.core.sample.filter.IBIZCPUSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"处理器" })
@RestController("DemoAPI-ibizcpu")
@RequestMapping("")
public class IBIZCPUResource {

    @Autowired
    public IIBIZCPUService ibizcpuService;

    @Autowired
    @Lazy
    public IBIZCPUMapping ibizcpuMapping;

    @PreAuthorize("hasPermission(this.ibizcpuMapping.toDomain(#ibizcpudto),'DemoSys-IBIZCPU-Create')")
    @ApiOperation(value = "新建处理器", tags = {"处理器" },  notes = "新建处理器")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizcpus")
    public ResponseEntity<IBIZCPUDTO> create(@Validated @RequestBody IBIZCPUDTO ibizcpudto) {
        IBIZCPU domain = ibizcpuMapping.toDomain(ibizcpudto);
		ibizcpuService.create(domain);
        IBIZCPUDTO dto = ibizcpuMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission(this.ibizcpuMapping.toDomain(#ibizcpudtos),'DemoSys-IBIZCPU-Create')")
    @ApiOperation(value = "批量新建处理器", tags = {"处理器" },  notes = "批量新建处理器")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizcpus/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<IBIZCPUDTO> ibizcpudtos) {
        ibizcpuService.createBatch(ibizcpuMapping.toDomain(ibizcpudtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @VersionCheck(entity = "ibizcpu" , versionfield = "updatedate")
    @PreAuthorize("hasPermission(this.ibizcpuService.get(#ibizcpu_id),'DemoSys-IBIZCPU-Update')")
    @ApiOperation(value = "更新处理器", tags = {"处理器" },  notes = "更新处理器")
	@RequestMapping(method = RequestMethod.PUT, value = "/ibizcpus/{ibizcpu_id}")
    public ResponseEntity<IBIZCPUDTO> update(@PathVariable("ibizcpu_id") String ibizcpu_id, @RequestBody IBIZCPUDTO ibizcpudto) {
		IBIZCPU domain  = ibizcpuMapping.toDomain(ibizcpudto);
        domain .setIbizcpuid(ibizcpu_id);
		ibizcpuService.update(domain );
		IBIZCPUDTO dto = ibizcpuMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission(this.ibizcpuService.getIbizcpuByEntities(this.ibizcpuMapping.toDomain(#ibizcpudtos)),'DemoSys-IBIZCPU-Update')")
    @ApiOperation(value = "批量更新处理器", tags = {"处理器" },  notes = "批量更新处理器")
	@RequestMapping(method = RequestMethod.PUT, value = "/ibizcpus/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<IBIZCPUDTO> ibizcpudtos) {
        ibizcpuService.updateBatch(ibizcpuMapping.toDomain(ibizcpudtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasPermission(this.ibizcpuService.get(#ibizcpu_id),'DemoSys-IBIZCPU-Remove')")
    @ApiOperation(value = "删除处理器", tags = {"处理器" },  notes = "删除处理器")
	@RequestMapping(method = RequestMethod.DELETE, value = "/ibizcpus/{ibizcpu_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("ibizcpu_id") String ibizcpu_id) {
         return ResponseEntity.status(HttpStatus.OK).body(ibizcpuService.remove(ibizcpu_id));
    }

    @PreAuthorize("hasPermission(this.ibizcpuService.getIbizcpuByIds(#ids),'DemoSys-IBIZCPU-Remove')")
    @ApiOperation(value = "批量删除处理器", tags = {"处理器" },  notes = "批量删除处理器")
	@RequestMapping(method = RequestMethod.DELETE, value = "/ibizcpus/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        ibizcpuService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PostAuthorize("hasPermission(this.ibizcpuMapping.toDomain(returnObject.body),'DemoSys-IBIZCPU-Get')")
    @ApiOperation(value = "获取处理器", tags = {"处理器" },  notes = "获取处理器")
	@RequestMapping(method = RequestMethod.GET, value = "/ibizcpus/{ibizcpu_id}")
    public ResponseEntity<IBIZCPUDTO> get(@PathVariable("ibizcpu_id") String ibizcpu_id) {
        IBIZCPU domain = ibizcpuService.get(ibizcpu_id);
        IBIZCPUDTO dto = ibizcpuMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取处理器草稿", tags = {"处理器" },  notes = "获取处理器草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/ibizcpus/getdraft")
    public ResponseEntity<IBIZCPUDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(ibizcpuMapping.toDto(ibizcpuService.getDraft(new IBIZCPU())));
    }

    @ApiOperation(value = "检查处理器", tags = {"处理器" },  notes = "检查处理器")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizcpus/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody IBIZCPUDTO ibizcpudto) {
        return  ResponseEntity.status(HttpStatus.OK).body(ibizcpuService.checkKey(ibizcpuMapping.toDomain(ibizcpudto)));
    }

    @PreAuthorize("hasPermission(this.ibizcpuMapping.toDomain(#ibizcpudto),'DemoSys-IBIZCPU-Save')")
    @ApiOperation(value = "保存处理器", tags = {"处理器" },  notes = "保存处理器")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizcpus/save")
    public ResponseEntity<Boolean> save(@RequestBody IBIZCPUDTO ibizcpudto) {
        return ResponseEntity.status(HttpStatus.OK).body(ibizcpuService.save(ibizcpuMapping.toDomain(ibizcpudto)));
    }

    @PreAuthorize("hasPermission(this.ibizcpuMapping.toDomain(#ibizcpudtos),'DemoSys-IBIZCPU-Save')")
    @ApiOperation(value = "批量保存处理器", tags = {"处理器" },  notes = "批量保存处理器")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizcpus/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<IBIZCPUDTO> ibizcpudtos) {
        ibizcpuService.saveBatch(ibizcpuMapping.toDomain(ibizcpudtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZCPU-searchDefault-all') and hasPermission(#context,'DemoSys-IBIZCPU-Get')")
	@ApiOperation(value = "获取DEFAULT", tags = {"处理器" } ,notes = "获取DEFAULT")
    @RequestMapping(method= RequestMethod.POST , value="/ibizcpus/fetchdefault")
	public ResponseEntity<List<IBIZCPUDTO>> fetchDefault(IBIZCPUSearchContext context) {
        Page<IBIZCPU> domains = ibizcpuService.searchDefault(context) ;
        List<IBIZCPUDTO> list = ibizcpuMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZCPU-searchDefault-all') and hasPermission(#context,'DemoSys-IBIZCPU-Get')")
	@ApiOperation(value = "查询DEFAULT", tags = {"处理器" } ,notes = "查询DEFAULT")
    @RequestMapping(method= RequestMethod.POST , value="/ibizcpus/searchdefault")
	public ResponseEntity<Page<IBIZCPUDTO>> searchDefault(@RequestBody IBIZCPUSearchContext context) {
        Page<IBIZCPU> domains = ibizcpuService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(ibizcpuMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}


}

