package cn.ibizlab.demoapi.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.demoapi.dto.*;
import cn.ibizlab.demoapi.mapping.*;
import cn.ibizlab.core.sample.domain.IBIZSupplier;
import cn.ibizlab.core.sample.service.IIBIZSupplierService;
import cn.ibizlab.core.sample.filter.IBIZSupplierSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"供货商" })
@RestController("DemoAPI-ibizsupplier")
@RequestMapping("")
public class IBIZSupplierResource {

    @Autowired
    public IIBIZSupplierService ibizsupplierService;

    @Autowired
    @Lazy
    public IBIZSupplierMapping ibizsupplierMapping;

    @PreAuthorize("hasPermission(this.ibizsupplierMapping.toDomain(#ibizsupplierdto),'DemoSys-IBIZSupplier-Create')")
    @ApiOperation(value = "新建供货商", tags = {"供货商" },  notes = "新建供货商")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizsuppliers")
    public ResponseEntity<IBIZSupplierDTO> create(@Validated @RequestBody IBIZSupplierDTO ibizsupplierdto) {
        IBIZSupplier domain = ibizsupplierMapping.toDomain(ibizsupplierdto);
		ibizsupplierService.create(domain);
        IBIZSupplierDTO dto = ibizsupplierMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission(this.ibizsupplierMapping.toDomain(#ibizsupplierdtos),'DemoSys-IBIZSupplier-Create')")
    @ApiOperation(value = "批量新建供货商", tags = {"供货商" },  notes = "批量新建供货商")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizsuppliers/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<IBIZSupplierDTO> ibizsupplierdtos) {
        ibizsupplierService.createBatch(ibizsupplierMapping.toDomain(ibizsupplierdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @VersionCheck(entity = "ibizsupplier" , versionfield = "updatetime")
    @PreAuthorize("hasPermission(this.ibizsupplierService.get(#ibizsupplier_id),'DemoSys-IBIZSupplier-Update')")
    @ApiOperation(value = "更新供货商", tags = {"供货商" },  notes = "更新供货商")
	@RequestMapping(method = RequestMethod.PUT, value = "/ibizsuppliers/{ibizsupplier_id}")
    public ResponseEntity<IBIZSupplierDTO> update(@PathVariable("ibizsupplier_id") String ibizsupplier_id, @RequestBody IBIZSupplierDTO ibizsupplierdto) {
		IBIZSupplier domain  = ibizsupplierMapping.toDomain(ibizsupplierdto);
        domain .setSupplierid(ibizsupplier_id);
		ibizsupplierService.update(domain );
		IBIZSupplierDTO dto = ibizsupplierMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission(this.ibizsupplierService.getIbizsupplierByEntities(this.ibizsupplierMapping.toDomain(#ibizsupplierdtos)),'DemoSys-IBIZSupplier-Update')")
    @ApiOperation(value = "批量更新供货商", tags = {"供货商" },  notes = "批量更新供货商")
	@RequestMapping(method = RequestMethod.PUT, value = "/ibizsuppliers/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<IBIZSupplierDTO> ibizsupplierdtos) {
        ibizsupplierService.updateBatch(ibizsupplierMapping.toDomain(ibizsupplierdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasPermission(this.ibizsupplierService.get(#ibizsupplier_id),'DemoSys-IBIZSupplier-Remove')")
    @ApiOperation(value = "删除供货商", tags = {"供货商" },  notes = "删除供货商")
	@RequestMapping(method = RequestMethod.DELETE, value = "/ibizsuppliers/{ibizsupplier_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("ibizsupplier_id") String ibizsupplier_id) {
         return ResponseEntity.status(HttpStatus.OK).body(ibizsupplierService.remove(ibizsupplier_id));
    }

    @PreAuthorize("hasPermission(this.ibizsupplierService.getIbizsupplierByIds(#ids),'DemoSys-IBIZSupplier-Remove')")
    @ApiOperation(value = "批量删除供货商", tags = {"供货商" },  notes = "批量删除供货商")
	@RequestMapping(method = RequestMethod.DELETE, value = "/ibizsuppliers/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        ibizsupplierService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PostAuthorize("hasPermission(this.ibizsupplierMapping.toDomain(returnObject.body),'DemoSys-IBIZSupplier-Get')")
    @ApiOperation(value = "获取供货商", tags = {"供货商" },  notes = "获取供货商")
	@RequestMapping(method = RequestMethod.GET, value = "/ibizsuppliers/{ibizsupplier_id}")
    public ResponseEntity<IBIZSupplierDTO> get(@PathVariable("ibizsupplier_id") String ibizsupplier_id) {
        IBIZSupplier domain = ibizsupplierService.get(ibizsupplier_id);
        IBIZSupplierDTO dto = ibizsupplierMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取供货商草稿", tags = {"供货商" },  notes = "获取供货商草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/ibizsuppliers/getdraft")
    public ResponseEntity<IBIZSupplierDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(ibizsupplierMapping.toDto(ibizsupplierService.getDraft(new IBIZSupplier())));
    }

    @ApiOperation(value = "检查供货商", tags = {"供货商" },  notes = "检查供货商")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizsuppliers/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody IBIZSupplierDTO ibizsupplierdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(ibizsupplierService.checkKey(ibizsupplierMapping.toDomain(ibizsupplierdto)));
    }

    @PreAuthorize("hasPermission(this.ibizsupplierMapping.toDomain(#ibizsupplierdto),'DemoSys-IBIZSupplier-Save')")
    @ApiOperation(value = "保存供货商", tags = {"供货商" },  notes = "保存供货商")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizsuppliers/save")
    public ResponseEntity<Boolean> save(@RequestBody IBIZSupplierDTO ibizsupplierdto) {
        return ResponseEntity.status(HttpStatus.OK).body(ibizsupplierService.save(ibizsupplierMapping.toDomain(ibizsupplierdto)));
    }

    @PreAuthorize("hasPermission(this.ibizsupplierMapping.toDomain(#ibizsupplierdtos),'DemoSys-IBIZSupplier-Save')")
    @ApiOperation(value = "批量保存供货商", tags = {"供货商" },  notes = "批量保存供货商")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizsuppliers/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<IBIZSupplierDTO> ibizsupplierdtos) {
        ibizsupplierService.saveBatch(ibizsupplierMapping.toDomain(ibizsupplierdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZSupplier-searchDefault-all') and hasPermission(#context,'DemoSys-IBIZSupplier-Get')")
	@ApiOperation(value = "获取DEFAULT", tags = {"供货商" } ,notes = "获取DEFAULT")
    @RequestMapping(method= RequestMethod.POST , value="/ibizsuppliers/fetchdefault")
	public ResponseEntity<List<IBIZSupplierDTO>> fetchDefault(IBIZSupplierSearchContext context) {
        Page<IBIZSupplier> domains = ibizsupplierService.searchDefault(context) ;
        List<IBIZSupplierDTO> list = ibizsupplierMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.CREATED)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZSupplier-searchDefault-all') and hasPermission(#context,'DemoSys-IBIZSupplier-Get')")
	@ApiOperation(value = "查询DEFAULT", tags = {"供货商" } ,notes = "查询DEFAULT")
    @RequestMapping(method= RequestMethod.POST , value="/ibizsuppliers/searchdefault")
	public ResponseEntity<Page<IBIZSupplierDTO>> searchDefault(@RequestBody IBIZSupplierSearchContext context) {
        Page<IBIZSupplier> domains = ibizsupplierService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.CREATED)
                .body(new PageImpl(ibizsupplierMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}


}

