package cn.ibizlab.demoapi.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.demoapi.dto.*;
import cn.ibizlab.demoapi.mapping.*;
import cn.ibizlab.core.sample.domain.IBIZStorage;
import cn.ibizlab.core.sample.service.IIBIZStorageService;
import cn.ibizlab.core.sample.filter.IBIZStorageSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"存储" })
@RestController("DemoAPI-ibizstorage")
@RequestMapping("")
public class IBIZStorageResource {

    @Autowired
    public IIBIZStorageService ibizstorageService;

    @Autowired
    @Lazy
    public IBIZStorageMapping ibizstorageMapping;

    @PreAuthorize("hasPermission(this.ibizstorageMapping.toDomain(#ibizstoragedto),'DemoSys-IBIZStorage-Create')")
    @ApiOperation(value = "新建存储", tags = {"存储" },  notes = "新建存储")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizstorages")
    public ResponseEntity<IBIZStorageDTO> create(@Validated @RequestBody IBIZStorageDTO ibizstoragedto) {
        IBIZStorage domain = ibizstorageMapping.toDomain(ibizstoragedto);
		ibizstorageService.create(domain);
        IBIZStorageDTO dto = ibizstorageMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission(this.ibizstorageMapping.toDomain(#ibizstoragedtos),'DemoSys-IBIZStorage-Create')")
    @ApiOperation(value = "批量新建存储", tags = {"存储" },  notes = "批量新建存储")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizstorages/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<IBIZStorageDTO> ibizstoragedtos) {
        ibizstorageService.createBatch(ibizstorageMapping.toDomain(ibizstoragedtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @VersionCheck(entity = "ibizstorage" , versionfield = "updatedate")
    @PreAuthorize("hasPermission(this.ibizstorageService.get(#ibizstorage_id),'DemoSys-IBIZStorage-Update')")
    @ApiOperation(value = "更新存储", tags = {"存储" },  notes = "更新存储")
	@RequestMapping(method = RequestMethod.PUT, value = "/ibizstorages/{ibizstorage_id}")
    public ResponseEntity<IBIZStorageDTO> update(@PathVariable("ibizstorage_id") String ibizstorage_id, @RequestBody IBIZStorageDTO ibizstoragedto) {
		IBIZStorage domain  = ibizstorageMapping.toDomain(ibizstoragedto);
        domain .setIbizstorageid(ibizstorage_id);
		ibizstorageService.update(domain );
		IBIZStorageDTO dto = ibizstorageMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission(this.ibizstorageService.getIbizstorageByEntities(this.ibizstorageMapping.toDomain(#ibizstoragedtos)),'DemoSys-IBIZStorage-Update')")
    @ApiOperation(value = "批量更新存储", tags = {"存储" },  notes = "批量更新存储")
	@RequestMapping(method = RequestMethod.PUT, value = "/ibizstorages/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<IBIZStorageDTO> ibizstoragedtos) {
        ibizstorageService.updateBatch(ibizstorageMapping.toDomain(ibizstoragedtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasPermission(this.ibizstorageService.get(#ibizstorage_id),'DemoSys-IBIZStorage-Remove')")
    @ApiOperation(value = "删除存储", tags = {"存储" },  notes = "删除存储")
	@RequestMapping(method = RequestMethod.DELETE, value = "/ibizstorages/{ibizstorage_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("ibizstorage_id") String ibizstorage_id) {
         return ResponseEntity.status(HttpStatus.OK).body(ibizstorageService.remove(ibizstorage_id));
    }

    @PreAuthorize("hasPermission(this.ibizstorageService.getIbizstorageByIds(#ids),'DemoSys-IBIZStorage-Remove')")
    @ApiOperation(value = "批量删除存储", tags = {"存储" },  notes = "批量删除存储")
	@RequestMapping(method = RequestMethod.DELETE, value = "/ibizstorages/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        ibizstorageService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PostAuthorize("hasPermission(this.ibizstorageMapping.toDomain(returnObject.body),'DemoSys-IBIZStorage-Get')")
    @ApiOperation(value = "获取存储", tags = {"存储" },  notes = "获取存储")
	@RequestMapping(method = RequestMethod.GET, value = "/ibizstorages/{ibizstorage_id}")
    public ResponseEntity<IBIZStorageDTO> get(@PathVariable("ibizstorage_id") String ibizstorage_id) {
        IBIZStorage domain = ibizstorageService.get(ibizstorage_id);
        IBIZStorageDTO dto = ibizstorageMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取存储草稿", tags = {"存储" },  notes = "获取存储草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/ibizstorages/getdraft")
    public ResponseEntity<IBIZStorageDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(ibizstorageMapping.toDto(ibizstorageService.getDraft(new IBIZStorage())));
    }

    @ApiOperation(value = "检查存储", tags = {"存储" },  notes = "检查存储")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizstorages/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody IBIZStorageDTO ibizstoragedto) {
        return  ResponseEntity.status(HttpStatus.OK).body(ibizstorageService.checkKey(ibizstorageMapping.toDomain(ibizstoragedto)));
    }

    @PreAuthorize("hasPermission(this.ibizstorageMapping.toDomain(#ibizstoragedto),'DemoSys-IBIZStorage-Save')")
    @ApiOperation(value = "保存存储", tags = {"存储" },  notes = "保存存储")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizstorages/save")
    public ResponseEntity<Boolean> save(@RequestBody IBIZStorageDTO ibizstoragedto) {
        return ResponseEntity.status(HttpStatus.OK).body(ibizstorageService.save(ibizstorageMapping.toDomain(ibizstoragedto)));
    }

    @PreAuthorize("hasPermission(this.ibizstorageMapping.toDomain(#ibizstoragedtos),'DemoSys-IBIZStorage-Save')")
    @ApiOperation(value = "批量保存存储", tags = {"存储" },  notes = "批量保存存储")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizstorages/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<IBIZStorageDTO> ibizstoragedtos) {
        ibizstorageService.saveBatch(ibizstorageMapping.toDomain(ibizstoragedtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZStorage-searchDefault-all') and hasPermission(#context,'DemoSys-IBIZStorage-Get')")
	@ApiOperation(value = "获取DEFAULT", tags = {"存储" } ,notes = "获取DEFAULT")
    @RequestMapping(method= RequestMethod.POST , value="/ibizstorages/fetchdefault")
	public ResponseEntity<List<IBIZStorageDTO>> fetchDefault(IBIZStorageSearchContext context) {
        Page<IBIZStorage> domains = ibizstorageService.searchDefault(context) ;
        List<IBIZStorageDTO> list = ibizstorageMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.CREATED)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZStorage-searchDefault-all') and hasPermission(#context,'DemoSys-IBIZStorage-Get')")
	@ApiOperation(value = "查询DEFAULT", tags = {"存储" } ,notes = "查询DEFAULT")
    @RequestMapping(method= RequestMethod.POST , value="/ibizstorages/searchdefault")
	public ResponseEntity<Page<IBIZStorageDTO>> searchDefault(@RequestBody IBIZStorageSearchContext context) {
        Page<IBIZStorage> domains = ibizstorageService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.CREATED)
                .body(new PageImpl(ibizstorageMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}


}

