import { Util, ViewTool } from '@/utils';
import { PanelDetailModel } from './panel-detail';

/**
 * 按钮模型
 *
 * @export
 * @class PanelButtonModel
 * @extends {PanelDetailModel}
 */
export class PanelButtonModel extends PanelDetailModel {

    /**
     * 按钮对应的界面行为
     *
     * @type {*}
     * @memberof PanelButtonModel
     */
    public uiAction: any;

    /**
     * 图标方向
     *
     * @type {('LEFT' | 'TOP' | 'RIGHT' | 'BOTTOM')}
     * @memberof PanelButtonModel
     */
    public iconAlign: 'LEFT' | 'TOP' | 'RIGHT' | 'BOTTOM';

    /**
     * 是否禁用
     *
     * @type {boolean}
     * @memberof PanelButtonModel
     */
    public disabled: boolean = false;

    /**
     * @description 按钮类型
     * @type {string}
     * @memberof PanelButtonModel
     */
    public buttonStyle: 'DEFAULT' | 'INVERSE' | 'PRIMARY' | 'INFO' | 'SUCCESS' | 'WARNING' | 'DANGER' | 'STYLE2' | 'STYLE3' | 'STYLE4';

    /**
     * @description 绘制模式
     * @type {string}
     * @memberof PanelButtonModel
     */
    public renderMode: 'BUTTON' | 'LINK';

    /**
     * 边框样式
     *
     * @type {('NONE' | 'SOLID' | 'DOTTED' | 'DASHED' | 'DOUBLE')}
     * @memberof PanelButtonModel
     */
    public borderStyle: 'NONE' | 'SOLID' | 'DOTTED' | 'DASHED' | 'DOUBLE' = 'NONE';

    /**
     * 数据部件名称
     *
     * @private
     * @type {string}
     * @memberof PanelButtonModel
     */
    private xDataControlName: string;

    /**
     * Creates an instance of PanelButtonModel.
     * @param {*} [opts={}]
     * @memberof PanelButtonModel
     */
    constructor(opts: any = {}) {
        super(opts);
        this.uiAction = opts.uiAction;
        this.buttonStyle = opts.buttonStyle;
        this.renderMode = opts.renderMode;
        this.iconAlign = opts.iconAlign || 'LEFT';
        this.disabled = opts.disabled;
        this.borderStyle = opts.borderStyle ? opts.borderStyle : 'NONE';
        this.xDataControlName = opts.xDataControlName;
    }

    /**
     * 获取元素样式（按钮元素，包含内容盒子 大小/边距/边框 的样式）
     *
     * @memberof PanelButtonModel
     */
    public getElementStyle() {
        const elementStyle = {};
        Object.assign(elementStyle, this.getBoxSizeStyle());
        Object.assign(elementStyle, this.getBoxSpacingStyle());
        Object.assign(elementStyle, this.getBoxBorderStyle());
        Object.assign(elementStyle, this.getBoxSelfAlignStyle());
        return elementStyle;
    }

    /**
     * 获取盒子边框样式（元素）
     *
     * @memberof PanelButtonModel
     */
    protected getBoxBorderStyle() {
        const boxStyle = {};
        if (this.borderStyle !== 'NONE') {
            switch (this.borderStyle) {
                // 实线边框
                case 'SOLID':
                    Object.assign(boxStyle, { 'border-style': 'solid' });
                    break;
                // 点状边框
                case 'DOTTED':
                    Object.assign(boxStyle, { 'border-style': 'dotted' });
                    break;
                // 虚线边框
                case 'DASHED':
                    Object.assign(boxStyle, { 'border-style': 'dashed' });
                    break;
                // 双线边框
                case 'DOUBLE':
                    Object.assign(boxStyle, { 'border-style': 'double' });
                    break;
                default:
                    console.warn(`${this.borderStyle}暂未支持`);
                    break;
            }
        }
        return boxStyle;
    }

    /**
     * 计算行为状态权限
     *
     * @private
     * @memberof PanelButtonModel
     */
    public computeActionAuthState() {
        if (this.uiAction && this.uiAction.dataaccaction && this.getData()) {
            let tempUIAction: any = Util.deepCopy(this.uiAction);
            ViewTool.calcActionItemAuthState(this.data, [tempUIAction], this.panel.appUIService ? this.panel.appUIService : null);
            this.visible = tempUIAction.visabled;
            this.disabled = tempUIAction.disabled;
        }
    }

    /**
     * 获取数据
     *
     * @private
     * @memberof PanelButtonModel
     */
    public getData() {
        if (this.parentName) {
            let parentItem = this.panel.layoutModelDetails[`${this.parentName}_${this.index}`];
            if (!parentItem) {
                parentItem = this.panel.layoutModelDetails[this.parentName];
            }
            if (parentItem && parentItem.getData()) {
                this.data = parentItem.getData();
            }
        }
        return this.data;
    }

    /**
     * 获取数据域
     *
     * @private
     * @memberof PanelButtonModel
     */
    public getDataArea() {
        let xData = null;
        let curLayoutModel = null;
        Object.values(this.panel.layoutModelDetails).forEach((layoutModel: any) => {
            if (layoutModel.name == this.name) {
                curLayoutModel = layoutModel;
            }
        })
        // 获取数据容器
        if (curLayoutModel) {
            const getDataArea = (cLayoutModel: any): any => {
                let dataArea = null;
                let parentLayoutModel = null;
                Object.values(this.panel.layoutModelDetails).forEach((pLayoutModel: any) => {
                    if (pLayoutModel.name == cLayoutModel.parentName) {
                        parentLayoutModel = pLayoutModel;
                        if (parentLayoutModel.dataRegionType == 'SINGLEDATA' || parentLayoutModel.dataRegionType == 'MULTIDATA') {
                            dataArea = parentLayoutModel;
                        }
                    }
                })
                if (!dataArea && parentLayoutModel) {
                    dataArea = getDataArea(parentLayoutModel);
                }
                return dataArea;
            }
            xData = getDataArea(curLayoutModel);
            // 获取激活部件
            if (!xData && this.xDataControlName) {
                Object.values(this.panel.layoutModelDetails).forEach((layoutModel: any) => {
                    if (layoutModel.name.toLowerCase() == this.xDataControlName.toLowerCase()) {
                        xData = layoutModel;
                        return
                    }
                })
            }
        }
        // 获取当前面板
        if (!xData) {
            xData = this.panel;
        }
        return xData;
    }

    /**
     * 加载
     *
     * @param {*} context
     * @param {*} viewParams
     * @memberof PanelButtonModel
     */
    public async load(context: any, viewParams: any) {
        this.computeActionAuthState();
    }
}