package cn.ibizlab.demoapi.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.demoapi.dto.*;
import cn.ibizlab.demoapi.mapping.*;
import cn.ibizlab.core.sample.domain.MicroComponent;
import cn.ibizlab.core.sample.service.IMicroComponentService;
import cn.ibizlab.core.sample.filter.MicroComponentSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"微组件" })
@RestController("DemoAPI-microcomponent")
@RequestMapping("")
public class MicroComponentResource {

    @Autowired
    public IMicroComponentService microcomponentService;

    @Autowired
    @Lazy
    public MicroComponentMapping microcomponentMapping;

    @PreAuthorize("hasPermission(this.microcomponentMapping.toDomain(#microcomponentdto),'DemoSys-MicroComponent-Create')")
    @ApiOperation(value = "新建微组件", tags = {"微组件" },  notes = "新建微组件")
	@RequestMapping(method = RequestMethod.POST, value = "/microcomponents")
    public ResponseEntity<MicroComponentDTO> create(@Validated @RequestBody MicroComponentDTO microcomponentdto) {
        MicroComponent domain = microcomponentMapping.toDomain(microcomponentdto);
		microcomponentService.create(domain);
        MicroComponentDTO dto = microcomponentMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission(this.microcomponentMapping.toDomain(#microcomponentdtos),'DemoSys-MicroComponent-Create')")
    @ApiOperation(value = "批量新建微组件", tags = {"微组件" },  notes = "批量新建微组件")
	@RequestMapping(method = RequestMethod.POST, value = "/microcomponents/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<MicroComponentDTO> microcomponentdtos) {
        microcomponentService.createBatch(microcomponentMapping.toDomain(microcomponentdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @VersionCheck(entity = "microcomponent" , versionfield = "updatedate")
    @PreAuthorize("hasPermission(this.microcomponentService.get(#microcomponent_id),'DemoSys-MicroComponent-Update')")
    @ApiOperation(value = "更新微组件", tags = {"微组件" },  notes = "更新微组件")
	@RequestMapping(method = RequestMethod.PUT, value = "/microcomponents/{microcomponent_id}")
    public ResponseEntity<MicroComponentDTO> update(@PathVariable("microcomponent_id") String microcomponent_id, @RequestBody MicroComponentDTO microcomponentdto) {
		MicroComponent domain  = microcomponentMapping.toDomain(microcomponentdto);
        domain .setMicrocomponentid(microcomponent_id);
		microcomponentService.update(domain );
		MicroComponentDTO dto = microcomponentMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission(this.microcomponentService.getMicrocomponentByEntities(this.microcomponentMapping.toDomain(#microcomponentdtos)),'DemoSys-MicroComponent-Update')")
    @ApiOperation(value = "批量更新微组件", tags = {"微组件" },  notes = "批量更新微组件")
	@RequestMapping(method = RequestMethod.PUT, value = "/microcomponents/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<MicroComponentDTO> microcomponentdtos) {
        microcomponentService.updateBatch(microcomponentMapping.toDomain(microcomponentdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasPermission(this.microcomponentService.get(#microcomponent_id),'DemoSys-MicroComponent-Remove')")
    @ApiOperation(value = "删除微组件", tags = {"微组件" },  notes = "删除微组件")
	@RequestMapping(method = RequestMethod.DELETE, value = "/microcomponents/{microcomponent_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("microcomponent_id") String microcomponent_id) {
         return ResponseEntity.status(HttpStatus.OK).body(microcomponentService.remove(microcomponent_id));
    }

    @PreAuthorize("hasPermission(this.microcomponentService.getMicrocomponentByIds(#ids),'DemoSys-MicroComponent-Remove')")
    @ApiOperation(value = "批量删除微组件", tags = {"微组件" },  notes = "批量删除微组件")
	@RequestMapping(method = RequestMethod.DELETE, value = "/microcomponents/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        microcomponentService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PostAuthorize("hasPermission(this.microcomponentMapping.toDomain(returnObject.body),'DemoSys-MicroComponent-Get')")
    @ApiOperation(value = "获取微组件", tags = {"微组件" },  notes = "获取微组件")
	@RequestMapping(method = RequestMethod.GET, value = "/microcomponents/{microcomponent_id}")
    public ResponseEntity<MicroComponentDTO> get(@PathVariable("microcomponent_id") String microcomponent_id) {
        MicroComponent domain = microcomponentService.get(microcomponent_id);
        MicroComponentDTO dto = microcomponentMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取微组件草稿", tags = {"微组件" },  notes = "获取微组件草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/microcomponents/getdraft")
    public ResponseEntity<MicroComponentDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(microcomponentMapping.toDto(microcomponentService.getDraft(new MicroComponent())));
    }

    @ApiOperation(value = "检查微组件", tags = {"微组件" },  notes = "检查微组件")
	@RequestMapping(method = RequestMethod.POST, value = "/microcomponents/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody MicroComponentDTO microcomponentdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(microcomponentService.checkKey(microcomponentMapping.toDomain(microcomponentdto)));
    }

    @PreAuthorize("hasPermission(this.microcomponentMapping.toDomain(#microcomponentdto),'DemoSys-MicroComponent-Save')")
    @ApiOperation(value = "保存微组件", tags = {"微组件" },  notes = "保存微组件")
	@RequestMapping(method = RequestMethod.POST, value = "/microcomponents/save")
    public ResponseEntity<Boolean> save(@RequestBody MicroComponentDTO microcomponentdto) {
        return ResponseEntity.status(HttpStatus.OK).body(microcomponentService.save(microcomponentMapping.toDomain(microcomponentdto)));
    }

    @PreAuthorize("hasPermission(this.microcomponentMapping.toDomain(#microcomponentdtos),'DemoSys-MicroComponent-Save')")
    @ApiOperation(value = "批量保存微组件", tags = {"微组件" },  notes = "批量保存微组件")
	@RequestMapping(method = RequestMethod.POST, value = "/microcomponents/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<MicroComponentDTO> microcomponentdtos) {
        microcomponentService.saveBatch(microcomponentMapping.toDomain(microcomponentdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-MicroComponent-searchDefault-all') and hasPermission(#context,'DemoSys-MicroComponent-Get')")
	@ApiOperation(value = "获取数据集", tags = {"微组件" } ,notes = "获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/microcomponents/fetchdefault")
	public ResponseEntity<List<MicroComponentDTO>> fetchDefault(MicroComponentSearchContext context) {
        Page<MicroComponent> domains = microcomponentService.searchDefault(context) ;
        List<MicroComponentDTO> list = microcomponentMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-MicroComponent-searchDefault-all') and hasPermission(#context,'DemoSys-MicroComponent-Get')")
	@ApiOperation(value = "查询数据集", tags = {"微组件" } ,notes = "查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/microcomponents/searchdefault")
	public ResponseEntity<Page<MicroComponentDTO>> searchDefault(@RequestBody MicroComponentSearchContext context) {
        Page<MicroComponent> domains = microcomponentService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(microcomponentMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}


}

