import { Environment } from '@/environments/environment';
import { UIActionTool, Util } from '@/utils';
import axios from 'axios';
import AppCenterService from "@service/app/app-center-service";
import UIService from '../ui-service';
import { Subject } from 'rxjs';
import IBIZCustomService from '@/service/ibizcustom/ibizcustom-service';
import IBIZCustomAuthService from '@/authservice/ibizcustom/ibizcustom-auth-service';
import CustomViewFormLoadUILogic from './custom-view-form-load-ui-logic';
import CustomViewGridLoadUILogic from './custom-view-grid-load-ui-logic';

/**
 * 自定义UI服务对象基类
 * 基于 APP/src/uiservice/%DE_PKGPATH%/%DE_PKGPATH%-ui-service-base.ts.ftl 生成
 * @export
 * @class IBIZCustomUIServiceBase
 */
export default class IBIZCustomUIServiceBase extends UIService {

    /**
     * 是否支持工作流
     * 
     * @memberof  IBIZCustomUIServiceBase
     */
    public isEnableWorkflow:boolean = false;

    /**
     * 是否支持实体主状态
     * 
     * @memberof  IBIZCustomUIServiceBase
     */
    public isEnableDEMainState:boolean = false;

    /**
     * 当前UI服务对应的数据服务对象
     * 
     * @memberof  IBIZCustomUIServiceBase
     */
    public dataService:IBIZCustomService = new IBIZCustomService();

    /**
     * 状态值
     * 
     * @memberof  IBIZCustomUIServiceBase
     */ 
    public stateValue: number = 0;

    /**
     * 状态属性
     * 
     * @memberof  IBIZCustomUIServiceBase
     */ 
    public stateField: string = "";

    /**
     * 主状态属性集合
     * 
     * @memberof  IBIZCustomUIServiceBase
     */  
    public mainStateFields:Array<any> = [];

    /**
     * 主状态集合Map
     * 
     * @memberof  IBIZCustomUIServiceBase
     */  
    public allDeMainStateMap:Map<string,string> = new Map();

    /**
     * 主状态操作标识Map
     * 
     * @memberof  IBIZCustomUIServiceBase
     */ 
    public allDeMainStateOPPrivsMap:Map<string,any> = new Map();

    /**
     * Creates an instance of  IBIZCustomUIServiceBase.
     * 
     * @param {*} [opts={}]
     * @memberof  IBIZCustomUIServiceBase
     */
    constructor(opts: any = {}) {
        super(opts);
        this.authService = new IBIZCustomAuthService(opts);
        this.initViewMap();
        this.initDeMainStateMap();
        this.initDeMainStateOPPrivsMap();
    }

    /**
     * 初始化视图Map
     * 
     * @memberof  IBIZCustomUIServiceBase
     */  
    public initViewMap(){
        this.allViewMap.set(':',{viewname:'customeditview',viewType:'DECUSTOMVIEW',srfappde:'ibizcustoms',component:'ibizcustom-custom-edit-view'});
        this.allViewMap.set(':',{viewname:'usr1118668829editview',viewType:'DEEDITVIEW',srfappde:'ibizcustoms',component:'ibizcustom-usr1118668829-edit-view'});
        this.allViewMap.set(':',{viewname:'customgridview',viewType:'DECUSTOMVIEW',srfappde:'ibizcustoms',component:'ibizcustom-custom-grid-view'});
    }

    /**
     * 初始化主状态集合
     * 
     * @memberof  IBIZCustomUIServiceBase
     */  
    public initDeMainStateMap(){
    }

    /**
     * 初始化主状态操作标识
     * 
     * @memberof  IBIZCustomUIServiceBase
     */  
    public initDeMainStateOPPrivsMap(){
    }


    /**
     * 获取指定数据的重定向页面
     * 
     * @param srfkey 数据主键
     * @param isEnableWorkflow  重定向视图是否需要处理流程中的数据
     * @param args  {context,viewParams} 用于计算流程   
     * @memberof  IBIZCustomUIServiceBase
     */
    public async getRDAppView(srfkey:string,isEnableWorkflow:boolean,args?:any){
        this.isEnableWorkflow = isEnableWorkflow;
        // 进行数据查询
        let result:any = await this.dataService.Get({ibizcustom:srfkey});
        const curData:any = result.data;
        //判断当前数据模式,默认为true，todo
        const iRealDEModel:boolean = true;
        let bDataInWF:boolean = false;
		let bWFMode:any = false;
		if (this.isEnableWorkflow) {
            return this.getWFView();
        } else {
            if ((args && args.viewParam && args.viewParam.srfwf && this.InWorkflowArray.indexOf(args.viewParam.srfwf) !== -1)) {
                return this.getWFView();
            }
        }
        let strPDTViewParam:string = await this.getDESDDEViewPDTParam(curData, bDataInWF, bWFMode);
        return this.allViewMap.get(strPDTViewParam);
    }

    /**
	 * 获取实际的数据类型
     * 
     * @memberof  IBIZCustomUIServiceBase
	 */
	public getRealDEType(entity:any){

    }

    /**
     * 获取实体单数据实体视图预定义参数
     * 
     * @param curData 当前数据
     * @param bDataInWF 是否有数据在工作流中
     * @param bWFMode   是否工作流模式
     * @memberof  IBIZCustomUIServiceBase
     */
    public async getDESDDEViewPDTParam(curData:any, bDataInWF:boolean, bWFMode:boolean){
        let strPDTParam:string = '';
		if (bDataInWF) {
			// 判断数据是否在流程中
        }
        //多表单，todo
        const multiFormDEField:string|null =null;

        if (multiFormDEField) {
			const objFormValue:string = curData[multiFormDEField];
			if(!Environment.isAppMode){
				return 'MOBEDITVIEW:'+objFormValue;
			}
			return 'EDITVIEW:'+objFormValue;
        }
		if(!Environment.isAppMode){
            if(this.getDEMainStateTag(curData)){
                return `MOBEDITVIEW:MSTAG:${ this.getDEMainStateTag(curData)}`;
            }
			return 'MOBEDITVIEW:';
        }
        if(this.getDEMainStateTag(curData)){
            return `EDITVIEW:MSTAG:${ this.getDEMainStateTag(curData)}`;
        }
		return 'EDITVIEW:';
    }

    /**
     * 获取数据对象的主状态标识
     * 
     * @param curData 当前数据
     * @memberof  IBIZCustomUIServiceBase
     */  
    public getDEMainStateTag(curData:any){
        if(this.mainStateFields.length === 0) return null;

        this.mainStateFields.forEach((singleMainField:any) =>{
            if(!(singleMainField in curData)){
                console.warn(`当前数据对象不包含属性「${singleMainField}」，根据「${singleMainField}」属性进行的主状态计算默认为空值`);
            }
        })
        for (let i = 0; i <= 1; i++) {
            let strTag:string = (curData[this.mainStateFields[0]] != null && curData[this.mainStateFields[0]] !== "")?(i == 0) ? `${curData[this.mainStateFields[0]]}` : "":"";
            if (this.mainStateFields.length >= 2) {
                for (let j = 0; j <= 1; j++) {
                    let strTag2:string = (curData[this.mainStateFields[1]] != null && curData[this.mainStateFields[1]] !== "")?`${strTag}__${(j == 0) ? `${curData[this.mainStateFields[1]]}` : ""}`:strTag;
                    if (this.mainStateFields.length >= 3) {
                        for (let k = 0; k <= 1; k++) {
                            let strTag3:string = (curData[this.mainStateFields[2]] != null && curData[this.mainStateFields[2]] !== "")?`${strTag2}__${(k == 0) ? `${curData[this.mainStateFields[2]]}` : ""}`:strTag2;
                            // 判断是否存在
                            return this.allDeMainStateMap.get(strTag3);
                        }
                    }else{
                        return this.allDeMainStateMap.get(strTag2);
                    }
                }
            }else{
                return this.allDeMainStateMap.get(strTag);
            }
        }
        return null;
    }

    /**
     * 获取数据对象当前操作标识
     * 
     * @param data 当前数据
     * @memberof  IBIZCustomUIServiceBase
     */  
    public getDEMainStateOPPrivs(data:any){
        if(this.getDEMainStateTag(data)){
            return this.allDeMainStateOPPrivsMap.get((this.getDEMainStateTag(data) as string));
        }else{
            return null;
        }
    }

    /**
     * 获取数据对象所有的操作标识
     * 
     * @param data 当前数据
     * @memberof  IBIZCustomUIServiceBase
     */ 
    public getAllOPPrivs(data:any){
       return this.authService.getOPPrivs(this.getDEMainStateOPPrivs(data));
    }

    /**
     * 执行界面逻辑
     *
     * @param {string} name
     * @param {any[]} args
     * @param {*} [context={}]
     * @param {*} [params={}]
     * @param {*} [$event]
     * @param {*} [xData]
     * @param {*} [actionContext]
     * @param {string} [srfParentDeName]
     * @memberof UIService
     */
    public async executeUILogic(name: string, args: any[], context?: any, params?: any, $event?: any, xData?: any, actionContext?: any, srfParentDeName?: string | undefined): Promise<any> {
        //  转化参数
        let data: any = null;
        if (args && Array.isArray(args) && args.length > 0) {
            data = args[0];
        } else {
            data = (Object.keys(args).length > 0) ? args : {};
        }
        if (name === 'CustomViewFormLoad') {
            return await CustomViewFormLoadUILogic.getInstance().execute(data, context, params, $event, xData, actionContext, srfParentDeName);
        }
        if (name === 'CustomViewGridLoad') {
            return await CustomViewGridLoadUILogic.getInstance().execute(data, context, params, $event, xData, actionContext, srfParentDeName);
        }
    } 

}