package cn.ibizlab.demoapi.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.demoapi.dto.*;
import cn.ibizlab.demoapi.mapping.*;
import cn.ibizlab.core.sample.domain.IBIZCustomer;
import cn.ibizlab.core.sample.service.IIBIZCustomerService;
import cn.ibizlab.core.sample.filter.IBIZCustomerSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"客户" })
@RestController("DemoAPI-ibizcustomer")
@RequestMapping("")
public class IBIZCustomerResource {

    @Autowired
    public IIBIZCustomerService ibizcustomerService;

    @Autowired
    @Lazy
    public IBIZCustomerMapping ibizcustomerMapping;

    @PreAuthorize("hasPermission(this.ibizcustomerMapping.toDomain(#ibizcustomerdto),'DemoSys-IBIZCustomer-Create')")
    @ApiOperation(value = "新建客户", tags = {"客户" },  notes = "新建客户")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizcustomers")
    public ResponseEntity<IBIZCustomerDTO> create(@Validated @RequestBody IBIZCustomerDTO ibizcustomerdto) {
        IBIZCustomer domain = ibizcustomerMapping.toDomain(ibizcustomerdto);
		ibizcustomerService.create(domain);
        IBIZCustomerDTO dto = ibizcustomerMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission(this.ibizcustomerMapping.toDomain(#ibizcustomerdtos),'DemoSys-IBIZCustomer-Create')")
    @ApiOperation(value = "批量新建客户", tags = {"客户" },  notes = "批量新建客户")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizcustomers/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<IBIZCustomerDTO> ibizcustomerdtos) {
        ibizcustomerService.createBatch(ibizcustomerMapping.toDomain(ibizcustomerdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PostAuthorize("hasPermission(this.ibizcustomerMapping.toDomain(returnObject.body),'DemoSys-IBIZCustomer-Get')")
    @ApiOperation(value = "获取客户", tags = {"客户" },  notes = "获取客户")
	@RequestMapping(method = RequestMethod.GET, value = "/ibizcustomers/{ibizcustomer_id}")
    public ResponseEntity<IBIZCustomerDTO> get(@PathVariable("ibizcustomer_id") String ibizcustomer_id) {
        IBIZCustomer domain = ibizcustomerService.get(ibizcustomer_id);
        IBIZCustomerDTO dto = ibizcustomerMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission(this.ibizcustomerService.get(#ibizcustomer_id),'DemoSys-IBIZCustomer-Remove')")
    @ApiOperation(value = "删除客户", tags = {"客户" },  notes = "删除客户")
	@RequestMapping(method = RequestMethod.DELETE, value = "/ibizcustomers/{ibizcustomer_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("ibizcustomer_id") String ibizcustomer_id) {
         return ResponseEntity.status(HttpStatus.OK).body(ibizcustomerService.remove(ibizcustomer_id));
    }

    @PreAuthorize("hasPermission(this.ibizcustomerService.getIbizcustomerByIds(#ids),'DemoSys-IBIZCustomer-Remove')")
    @ApiOperation(value = "批量删除客户", tags = {"客户" },  notes = "批量删除客户")
	@RequestMapping(method = RequestMethod.DELETE, value = "/ibizcustomers/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        ibizcustomerService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @VersionCheck(entity = "ibizcustomer" , versionfield = "updatedate")
    @PreAuthorize("hasPermission(this.ibizcustomerService.get(#ibizcustomer_id),'DemoSys-IBIZCustomer-Update')")
    @ApiOperation(value = "更新客户", tags = {"客户" },  notes = "更新客户")
	@RequestMapping(method = RequestMethod.PUT, value = "/ibizcustomers/{ibizcustomer_id}")
    public ResponseEntity<IBIZCustomerDTO> update(@PathVariable("ibizcustomer_id") String ibizcustomer_id, @RequestBody IBIZCustomerDTO ibizcustomerdto) {
		IBIZCustomer domain  = ibizcustomerMapping.toDomain(ibizcustomerdto);
        domain .setIbizcustomerid(ibizcustomer_id);
		ibizcustomerService.update(domain );
		IBIZCustomerDTO dto = ibizcustomerMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission(this.ibizcustomerService.getIbizcustomerByEntities(this.ibizcustomerMapping.toDomain(#ibizcustomerdtos)),'DemoSys-IBIZCustomer-Update')")
    @ApiOperation(value = "批量更新客户", tags = {"客户" },  notes = "批量更新客户")
	@RequestMapping(method = RequestMethod.PUT, value = "/ibizcustomers/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<IBIZCustomerDTO> ibizcustomerdtos) {
        ibizcustomerService.updateBatch(ibizcustomerMapping.toDomain(ibizcustomerdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "检查客户", tags = {"客户" },  notes = "检查客户")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizcustomers/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody IBIZCustomerDTO ibizcustomerdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(ibizcustomerService.checkKey(ibizcustomerMapping.toDomain(ibizcustomerdto)));
    }

    @ApiOperation(value = "获取客户草稿", tags = {"客户" },  notes = "获取客户草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/ibizcustomers/getdraft")
    public ResponseEntity<IBIZCustomerDTO> getDraft(IBIZCustomerDTO dto) {
        IBIZCustomer domain = ibizcustomerMapping.toDomain(dto);
        return ResponseEntity.status(HttpStatus.OK).body(ibizcustomerMapping.toDto(ibizcustomerService.getDraft(domain)));
    }

    @PreAuthorize("hasPermission(this.ibizcustomerMapping.toDomain(#ibizcustomerdto),'DemoSys-IBIZCustomer-Save')")
    @ApiOperation(value = "保存客户", tags = {"客户" },  notes = "保存客户")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizcustomers/save")
    public ResponseEntity<IBIZCustomerDTO> save(@RequestBody IBIZCustomerDTO ibizcustomerdto) {
        IBIZCustomer domain = ibizcustomerMapping.toDomain(ibizcustomerdto);
        ibizcustomerService.save(domain);
        return ResponseEntity.status(HttpStatus.OK).body(ibizcustomerMapping.toDto(domain));
    }

    @PreAuthorize("hasPermission(this.ibizcustomerMapping.toDomain(#ibizcustomerdtos),'DemoSys-IBIZCustomer-Save')")
    @ApiOperation(value = "批量保存客户", tags = {"客户" },  notes = "批量保存客户")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizcustomers/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<IBIZCustomerDTO> ibizcustomerdtos) {
        ibizcustomerService.saveBatch(ibizcustomerMapping.toDomain(ibizcustomerdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZCustomer-searchDefault-all') and hasPermission(#context,'DemoSys-IBIZCustomer-Get')")
	@ApiOperation(value = "获取DEFAULT", tags = {"客户" } ,notes = "获取DEFAULT")
    @RequestMapping(method= RequestMethod.GET , value="/ibizcustomers/fetchdefault")
	public ResponseEntity<List<IBIZCustomerDTO>> fetchDefault(IBIZCustomerSearchContext context) {
        Page<IBIZCustomer> domains = ibizcustomerService.searchDefault(context) ;
        List<IBIZCustomerDTO> list = ibizcustomerMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZCustomer-searchDefault-all') and hasPermission(#context,'DemoSys-IBIZCustomer-Get')")
	@ApiOperation(value = "查询DEFAULT", tags = {"客户" } ,notes = "查询DEFAULT")
    @RequestMapping(method= RequestMethod.POST , value="/ibizcustomers/searchdefault")
	public ResponseEntity<Page<IBIZCustomerDTO>> searchDefault(@RequestBody IBIZCustomerSearchContext context) {
        Page<IBIZCustomer> domains = ibizcustomerService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(ibizcustomerMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}



}

