import { Http } from '@/utils';
import { Subscription } from 'rxjs';

/**
 * 按钮loadding状态服务
 *
 * @export
 * @class ButtonLoadingService
 */
export class ButtonLoadingService {

    /**
     * 单例变量声明
     *
     * @private
     * @static
     * @type {ButtonLoadingService}
     * @memberof ButtonLoadingService
     */
    private static ButtonLoadingService: ButtonLoadingService;

    /**
     * loadding状态事件
     *
     * @public
     * @type {(Subscription | undefined)}
     * @memberof ButtonLoadingService
     */
    public buttonLoadingStateEvent: Subscription | null = null; 

    /**
     * @description dom映射对象
     * @type {Map<string, any>}
     * @memberof ButtonLoadingService
     */
    public domMap: Map<string, any> = new Map();

    /**
     * 获取 ButtonLoadingService 单例对象
     *
     * @static
     * @returns {ButtonLoadingService}
     * @memberof ButtonLoadingService
     */
    public static getInstance(): ButtonLoadingService {
        if (!ButtonLoadingService.ButtonLoadingService) {
            ButtonLoadingService.ButtonLoadingService = new ButtonLoadingService();
        }
        return this.ButtonLoadingService;
    }

    /**
     * 初始化指令所绑定的元素状态
     *
     * @param {any} el 指令所绑定的元素
     * @param {any} binding 指令附加参数
     * @memberof  ButtonLoadingService
     */
    public initElement(el:any, binding:any){
        if(binding && binding.arg){
            // 工具栏按钮
            if(Object.is(binding.arg,'i-button')){
                if(el.getElementsByTagName('i') && el.getElementsByTagName('i').length >0){
                    let iconElement:any = el.getElementsByTagName('i')[0];
                    iconElement.setAttribute('ownclassname',iconElement.className);
                    const uuid: string = String(el.__vue__._uid);
                    el.setAttribute('uuid',uuid);
                    this.domMap.set(uuid, iconElement);
                } 
            }
        }
    }

    /**
     * 设置loadding状态
     *
     * @param {any} el 指令所绑定的元素
     * @param {any} binding 指令附加参数
     * @memberof  ButtonLoadingService
     */
    public setLoadState(el:any, binding:any){
      if (!this.buttonLoadingStateEvent) {
        this.buttonLoadingStateEvent = Http.getInstance().getNotifyObject().subscribe((result:any) =>{
          if(result && result.action && Object.is(result.action,'setloadstate')){
              if(result && result.state){
                  this.addLoadState();
              }else{
                  this.removeLoadState();
              }
          }
        })
      }
    }

    /**
     * 添加loadding状态
     *
     * @param {any} el 指令所绑定的元素
     * @param {any} binding 指令附加参数
     * @memberof  ButtonLoadingService
     */
    public addLoadState(){
        this.domMap.forEach((dom: any, key: string) => {
          dom.className = "el-icon-loading";
        })
    }

    /**
     * 移除loadding状态
     *
     * @param {any} el 指令所绑定的元素
     * @param {any} binding 指令附加参数
     * @memberof  ButtonLoadingService
     */
    public removeLoadState(){
        this.domMap.forEach((dom: any, key: string) => {
          dom.className = dom.getAttribute('ownclassname');
        })
    }

    /**
     * 清除资源（取消订阅）
     *
     * @param {any} el 指令所绑定的元素
     * @param {any} binding 指令附加参数
     * @memberof  ButtonLoadingService
     */
    public clearResource(el:any, binding:any){
        if(this.buttonLoadingStateEvent && this.domMap.size == 0){
            this.buttonLoadingStateEvent.unsubscribe();
            this.buttonLoadingStateEvent = null;
        }
        const uuid: string = el.getAttribute('uuid');
        if (uuid && this.domMap.has(uuid)) {
          this.domMap.delete(uuid);
        }
    }
   

}
export  default {
    bind(el:any, binding:any) {
        ButtonLoadingService.getInstance().initElement(el, binding);
    },
    inserted(el:any, binding:any) {
        ButtonLoadingService.getInstance().setLoadState(el, binding);
    },
    unbind(el:any, binding:any) {
        ButtonLoadingService.getInstance().clearResource(el,binding);
    }
}