package cn.ibizlab.demoapi.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.demoapi.dto.*;
import cn.ibizlab.demoapi.mapping.*;
import cn.ibizlab.core.sample.domain.IBIZAccount;
import cn.ibizlab.core.sample.service.IIBIZAccountService;
import cn.ibizlab.core.sample.filter.IBIZAccountSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"账户" })
@RestController("DemoAPI-ibizaccount")
@RequestMapping("")
public class IBIZAccountResource {

    @Autowired
    public IIBIZAccountService ibizaccountService;

    @Autowired
    @Lazy
    public IBIZAccountMapping ibizaccountMapping;

    @PreAuthorize("hasPermission(this.ibizaccountMapping.toDomain(#ibizaccountdto),'DemoSys-IBIZAccount-Create')")
    @ApiOperation(value = "新建账户", tags = {"账户" },  notes = "新建账户")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizaccounts")
    public ResponseEntity<IBIZAccountDTO> create(@Validated @RequestBody IBIZAccountDTO ibizaccountdto) {
        IBIZAccount domain = ibizaccountMapping.toDomain(ibizaccountdto);
		ibizaccountService.create(domain);
        IBIZAccountDTO dto = ibizaccountMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission(this.ibizaccountMapping.toDomain(#ibizaccountdtos),'DemoSys-IBIZAccount-Create')")
    @ApiOperation(value = "批量新建账户", tags = {"账户" },  notes = "批量新建账户")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizaccounts/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<IBIZAccountDTO> ibizaccountdtos) {
        ibizaccountService.createBatch(ibizaccountMapping.toDomain(ibizaccountdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @VersionCheck(entity = "ibizaccount" , versionfield = "updatedate")
    @PreAuthorize("hasPermission(this.ibizaccountService.get(#ibizaccount_id),'DemoSys-IBIZAccount-Update')")
    @ApiOperation(value = "更新账户", tags = {"账户" },  notes = "更新账户")
	@RequestMapping(method = RequestMethod.PUT, value = "/ibizaccounts/{ibizaccount_id}")
    public ResponseEntity<IBIZAccountDTO> update(@PathVariable("ibizaccount_id") String ibizaccount_id, @RequestBody IBIZAccountDTO ibizaccountdto) {
		IBIZAccount domain  = ibizaccountMapping.toDomain(ibizaccountdto);
        domain .setIbizaccountid(ibizaccount_id);
		ibizaccountService.update(domain );
		IBIZAccountDTO dto = ibizaccountMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission(this.ibizaccountService.getIbizaccountByEntities(this.ibizaccountMapping.toDomain(#ibizaccountdtos)),'DemoSys-IBIZAccount-Update')")
    @ApiOperation(value = "批量更新账户", tags = {"账户" },  notes = "批量更新账户")
	@RequestMapping(method = RequestMethod.PUT, value = "/ibizaccounts/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<IBIZAccountDTO> ibizaccountdtos) {
        ibizaccountService.updateBatch(ibizaccountMapping.toDomain(ibizaccountdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasPermission(this.ibizaccountService.get(#ibizaccount_id),'DemoSys-IBIZAccount-Remove')")
    @ApiOperation(value = "删除账户", tags = {"账户" },  notes = "删除账户")
	@RequestMapping(method = RequestMethod.DELETE, value = "/ibizaccounts/{ibizaccount_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("ibizaccount_id") String ibizaccount_id) {
         return ResponseEntity.status(HttpStatus.OK).body(ibizaccountService.remove(ibizaccount_id));
    }

    @PreAuthorize("hasPermission(this.ibizaccountService.getIbizaccountByIds(#ids),'DemoSys-IBIZAccount-Remove')")
    @ApiOperation(value = "批量删除账户", tags = {"账户" },  notes = "批量删除账户")
	@RequestMapping(method = RequestMethod.DELETE, value = "/ibizaccounts/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        ibizaccountService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PostAuthorize("hasPermission(this.ibizaccountMapping.toDomain(returnObject.body),'DemoSys-IBIZAccount-Get')")
    @ApiOperation(value = "获取账户", tags = {"账户" },  notes = "获取账户")
	@RequestMapping(method = RequestMethod.GET, value = "/ibizaccounts/{ibizaccount_id}")
    public ResponseEntity<IBIZAccountDTO> get(@PathVariable("ibizaccount_id") String ibizaccount_id) {
        IBIZAccount domain = ibizaccountService.get(ibizaccount_id);
        IBIZAccountDTO dto = ibizaccountMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取账户草稿", tags = {"账户" },  notes = "获取账户草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/ibizaccounts/getdraft")
    public ResponseEntity<IBIZAccountDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(ibizaccountMapping.toDto(ibizaccountService.getDraft(new IBIZAccount())));
    }

    @ApiOperation(value = "检查账户", tags = {"账户" },  notes = "检查账户")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizaccounts/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody IBIZAccountDTO ibizaccountdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(ibizaccountService.checkKey(ibizaccountMapping.toDomain(ibizaccountdto)));
    }

    @PreAuthorize("hasPermission(this.ibizaccountMapping.toDomain(#ibizaccountdto),'DemoSys-IBIZAccount-Save')")
    @ApiOperation(value = "保存账户", tags = {"账户" },  notes = "保存账户")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizaccounts/save")
    public ResponseEntity<Boolean> save(@RequestBody IBIZAccountDTO ibizaccountdto) {
        return ResponseEntity.status(HttpStatus.OK).body(ibizaccountService.save(ibizaccountMapping.toDomain(ibizaccountdto)));
    }

    @PreAuthorize("hasPermission(this.ibizaccountMapping.toDomain(#ibizaccountdtos),'DemoSys-IBIZAccount-Save')")
    @ApiOperation(value = "批量保存账户", tags = {"账户" },  notes = "批量保存账户")
	@RequestMapping(method = RequestMethod.POST, value = "/ibizaccounts/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<IBIZAccountDTO> ibizaccountdtos) {
        ibizaccountService.saveBatch(ibizaccountMapping.toDomain(ibizaccountdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZAccount-searchDefault-all') and hasPermission(#context,'DemoSys-IBIZAccount-Get')")
	@ApiOperation(value = "获取DEFAULT", tags = {"账户" } ,notes = "获取DEFAULT")
    @RequestMapping(method= RequestMethod.POST , value="/ibizaccounts/fetchdefault")
	public ResponseEntity<List<IBIZAccountDTO>> fetchDefault(IBIZAccountSearchContext context) {
        Page<IBIZAccount> domains = ibizaccountService.searchDefault(context) ;
        List<IBIZAccountDTO> list = ibizaccountMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZAccount-searchDefault-all') and hasPermission(#context,'DemoSys-IBIZAccount-Get')")
	@ApiOperation(value = "查询DEFAULT", tags = {"账户" } ,notes = "查询DEFAULT")
    @RequestMapping(method= RequestMethod.POST , value="/ibizaccounts/searchdefault")
	public ResponseEntity<Page<IBIZAccountDTO>> searchDefault(@RequestBody IBIZAccountSearchContext context) {
        Page<IBIZAccount> domains = ibizaccountService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(ibizaccountMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}


}

