package cn.ibizlab.demoapi.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.demoapi.dto.*;
import cn.ibizlab.demoapi.mapping.*;
import cn.ibizlab.core.sample.domain.IBIZCPU;
import cn.ibizlab.core.sample.service.IBIZCPUService;
import cn.ibizlab.core.sample.filter.IBIZCPUSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"处理器" })
@RestController("demoapi-ibizcpu")
@RequestMapping("${ibiz.rest.basePath.demoapi:/demoapi/}")
public class IBIZCPUResource {

    @Autowired
    public IBIZCPUService ibizcpuService;

    @Autowired
    @Lazy
    public IBIZCPUDTOMapping ibizcpudtoMapping;


    @ApiOperation(value = "创建Create", tags = {"处理器" },  notes = "IBIZCPU-Create")
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZCPU-Create-all') or hasPermission(this.ibizcpudtoMapping.toDomain(#dto),'DemoSys-IBIZCPU-Create')")
    @RequestMapping(method = RequestMethod.POST, value = "ibizcpus")
    public ResponseEntity<IBIZCPUDTO> create
            (@Validated @RequestBody IBIZCPUDTO dto) {
        IBIZCPU domain = ibizcpudtoMapping.toDomain(dto);
        ibizcpuService.create(domain);
        IBIZCPU rt = domain;
        return ResponseEntity.status(HttpStatus.OK).body(ibizcpudtoMapping.toDto(rt));
    }

    @ApiOperation(value = "获取Get", tags = {"处理器" },  notes = "IBIZCPU-Get")
    @PostAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZCPU-Get-all') or hasPermission(this.ibizcpudtoMapping.toDomain(returnObject.body),'DemoSys-IBIZCPU-Get')")
    @RequestMapping(method = RequestMethod.GET, value = "ibizcpus/{ibizcpuid}")
    public ResponseEntity<IBIZCPUDTO> getByIbizcpuid
            (@PathVariable("ibizcpuid") String ibizcpuid) {
        IBIZCPU rt = ibizcpuService.get(ibizcpuid);
        return ResponseEntity.status(HttpStatus.OK).body(ibizcpudtoMapping.toDto(rt));
    }

    @ApiOperation(value = "删除Remove", tags = {"处理器" },  notes = "IBIZCPU-Remove")
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZCPU-Remove-all') or hasPermission(this.ibizcpuService.get(#ibizcpuid),'DemoSys-IBIZCPU-Remove')")
    @RequestMapping(method = RequestMethod.DELETE, value = "ibizcpus/{ibizcpuid}")
    public ResponseEntity<Boolean> removeByIbizcpuid
            (@PathVariable("ibizcpuid") String ibizcpuid) {
        Boolean rt = ibizcpuService.remove(ibizcpuid);
        return ResponseEntity.status(HttpStatus.OK).body(rt);
    }

    @ApiOperation(value = "更新Update", tags = {"处理器" },  notes = "IBIZCPU-Update")
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZCPU-Update-all') or hasPermission(this.ibizcpuService.get(#ibizcpuid),'DemoSys-IBIZCPU-Update')")
    @VersionCheck(entity = "ibizcpu" , versionfield = "updateDate")
    @RequestMapping(method = RequestMethod.PUT, value = "ibizcpus/{ibizcpuid}")
    public ResponseEntity<IBIZCPUDTO> updateByIbizcpuid
            (@PathVariable("ibizcpuid") String ibizcpuid, @Validated @RequestBody IBIZCPUDTO dto) {
        IBIZCPU domain = ibizcpudtoMapping.toDomain(dto);
        domain.setIbizcpuid(ibizcpuid);
        ibizcpuService.update(domain);
        IBIZCPU rt = domain;
        return ResponseEntity.status(HttpStatus.OK).body(ibizcpudtoMapping.toDto(rt));
    }

    @ApiOperation(value = "校验CheckKey", tags = {"处理器" },  notes = "IBIZCPU-CheckKey")
    @RequestMapping(method = RequestMethod.POST, value = "ibizcpus/checkkey")
    public ResponseEntity<Integer> checkKey
            (@Validated @RequestBody IBIZCPUDTO dto) {
        IBIZCPU domain = ibizcpudtoMapping.toDomain(dto);
        Integer rt = ibizcpuService.checkKey(domain);
        return ResponseEntity.status(HttpStatus.OK).body(rt);
    }

    @ApiOperation(value = "草稿GetDraft", tags = {"处理器" },  notes = "IBIZCPU-GetDraft")
    @RequestMapping(method = RequestMethod.GET, value = "ibizcpus/getdraft")
    public ResponseEntity<IBIZCPUDTO> getDraft
            (IBIZCPUDTO dto) {
        IBIZCPU domain = ibizcpudtoMapping.toDomain(dto);
        IBIZCPU rt = ibizcpuService.getDraft(domain);
        return ResponseEntity.status(HttpStatus.OK).body(ibizcpudtoMapping.toDto(rt));
    }

    @ApiOperation(value = "保存Save", tags = {"处理器" },  notes = "IBIZCPU-Save")
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZCPU-Save-all') or hasPermission(this.ibizcpudtoMapping.toDomain(#dto),'DemoSys-IBIZCPU-Save')")
    @RequestMapping(method = RequestMethod.POST, value = "ibizcpus/save")
    public ResponseEntity<IBIZCPUDTO> save
            (@Validated @RequestBody IBIZCPUDTO dto) {
        IBIZCPU domain = ibizcpudtoMapping.toDomain(dto);
        ibizcpuService.save(domain);
        IBIZCPU rt = domain;
        return ResponseEntity.status(HttpStatus.OK).body(ibizcpudtoMapping.toDto(rt));
    }

    @ApiOperation(value = "查询FetchDefault", tags = {"处理器" },  notes = "IBIZCPU-FetchDefault")
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','DemoSys-IBIZCPU-searchDefault-all','DemoSys-IBIZCPU-Get-all') or hasPermission(#dto,'DemoSys-IBIZCPU-Get')")
    @RequestMapping(method = RequestMethod.GET, value = "ibizcpus/fetchdefault")
    public ResponseEntity<List<IBIZCPUDTO>> fetchDefault
            (IBIZCPUSearchContext dto) {
        Page<IBIZCPU> domains = ibizcpuService.searchDefault(dto) ;
        List<IBIZCPUDTO> list = ibizcpudtoMapping.toDto(domains.getContent());
            return ResponseEntity.status(HttpStatus.OK)
            .header("x-page", String.valueOf(dto.getPageable().getPageNumber()))
            .header("x-per-page", String.valueOf(dto.getPageable().getPageSize()))
            .header("x-total", String.valueOf(domains.getTotalElements()))
            .body(list);
    }


    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','demo-IBIZCPU-Create-all')")
    @ApiOperation(value = "批量新建处理器", tags = {"处理器" },  notes = "批量新建处理器")
	@RequestMapping(method = RequestMethod.POST, value = "ibizcpus/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<IBIZCPUDTO> dtos) {
        ibizcpuService.createBatch(ibizcpudtoMapping.toDomain(dtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','demo-IBIZCPU-Update-all')")
    @ApiOperation(value = "批量更新处理器", tags = {"处理器" },  notes = "批量更新处理器")
	@RequestMapping(method = RequestMethod.PUT, value = "ibizcpus/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<IBIZCPUDTO> dtos) {
        ibizcpuService.updateBatch(ibizcpudtoMapping.toDomain(dtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','demo-IBIZCPU-Save-all')")
    @ApiOperation(value = "批量保存处理器", tags = {"处理器" },  notes = "批量保存处理器")
	@RequestMapping(method = RequestMethod.POST, value = "ibizcpus/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<IBIZCPUDTO> dtos) {
        ibizcpuService.saveBatch(ibizcpudtoMapping.toDomain(dtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','demo-IBIZCPU-Save-all')")
    @ApiOperation(value = "批量导入处理器", tags = {"处理器" },  notes = "批量导入处理器")
	@RequestMapping(method = RequestMethod.POST, value = "ibizcpus/import")
    public ResponseEntity<Map> importData(@RequestParam(value = "config") String config ,@RequestBody List<IBIZCPUDTO> dtos) {
        Map rs=new HashMap();
        if(ObjectUtils.isEmpty(dtos)){
            rs.put("rst", 1);
            rs.put("msg", "未传入内容");
            return ResponseEntity.status(HttpStatus.NO_CONTENT).body(rs);
        }
        else{
            List<IBIZCPU> list=ibizcpudtoMapping.toDomain(dtos);
            List<IBIZCPU> rtList=new ArrayList();
            list.forEach(item->{
                if(ibizcpuService.save(item))
                    rtList.add(item);
            });
            rs.put("rst", 0);
            rs.put("data",rtList);
        }
        return  ResponseEntity.status(HttpStatus.OK).body(rs);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','demo-IBIZCPU-Remove-all')")
    @ApiOperation(value = "批量删除处理器", tags = {"处理器" },  notes = "批量删除处理器")
	@RequestMapping(method = RequestMethod.DELETE, value = {"ibizcpus","/ibizcpus/batch"})
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        ibizcpuService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }
}
