import { Util } from "@/utils";

/**
 * 面板成员模型
 *
 * @export
 * @class PanelDetailModel
 */
export class PanelDetailModel {

    /**
     * 类型 视图布局/项布局
     *
     * @type {('VIEWLAYOUT' | 'ITEMLAYOUT')}
     * @memberof PanelDetailModel
     */
    public type: 'VIEWLAYOUT' | 'ITEMLAYOUT' = 'VIEWLAYOUT';

    /**
     * 成员标题
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public caption: string = '';

    /**
     * 成员类型
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public itemType: string = '';

    /**
     * 成员样式
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public itemStyle: string = '';

    /**
     * 面板对象
     *
     * @type {*}
     * @memberof PanelDetailModel
     */
    public panel: any = null;

    /**
     * 成员名称
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public name: string = '';

    /**
     * 成员是否显示
     *
     * @type {boolean}
     * @memberof PanelDetailModel
     */
    public visible: boolean = true;

    /**
     * 当前数据
     *
     * @type {*}
     * @memberof PanelDetailModel
     */
    protected data: any;

    /**
     * 布局模式
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public layout: string = '';

    /**
     *  布局占位（边缘布局使用）
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public layoutPos: string = '';

    /**
     * 布局高度
     *
     * @type {number}
     * @memberof PanelDetailModel
     */
    public layoutHeight: number = 0;

    /**
     * 高度模式
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public heightMode: string = '';

    /**
     * 布局宽度
     *
     * @type {number}
     * @memberof PanelDetailModel
     */
    public layoutWidth: number = 0;

    /**
     * 宽度模式
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public widthMode: string = '';

    /**
     * 下档间隔
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public spacingBottom: string = '';

    /**
     * 左侧间隔
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public spacingLeft: string = '';

    /**
     * 右侧间隔
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public spacingRight: string = '';

    /**
     * 顶部间隔
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public spacingTop: string = '';

    /**
     * 自身水平对齐模式
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public hAlignSelf: string = '';

    /**
     * 自身垂直对齐模式
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public vAlignSelf: string = '';

    /**
     * Flex延伸
     *
     * @type {number}
     * @memberof PanelDetailModel
     */
    public flexGrow: number = 0;

    /**
     * flex布局参数
     *
     * @type {*}
     * @memberof PanelDetailModel
     */
    public flexParams: any = {};

    /**
     * 是否显示标题
     *
     * @type {boolean}
     * @memberof PanelDetailModel
     */
    public isShowCaption: boolean = false;

    /**
     * 界面样式表
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public sysCss: string = ''

    /**
     * @description 图标
     * @type {*}
     * @memberof PanelDetailModel
     */
    public sysImage: any = {};

    /**
     * 预置类型
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public predefinedType: string = '';

    /**
     * 下标
     *
     * @type {number}
     * @memberof PanelDetailModel
     */
    protected index: number = 0;

    /**
     * 父项名称
     *
     * @type {string | undefined}
     * @memberof PanelDetailModel
     */
    public parentName: string | undefined = undefined;

    /**
     * Creates an instance of PanelDetailModel.
     * PanelDetailModel 实例
     * 
     * @param {*} [opts={}]
     * @memberof PanelDetailModel
     */
    constructor(opts: any = {}) {
        this.type = opts.type;
        this.caption = opts.caption;
        this.itemType = opts.itemType;
        this.panel = opts.panel;
        this.name = opts.name;
        this.visible = opts.visible;
        this.layout = opts.layout;
        this.layoutPos = opts.layoutPos;
        this.layoutHeight = opts.layoutHeight;
        this.heightMode = opts.heightMode;
        this.layoutWidth = opts.layoutWidth;
        this.widthMode = opts.widthMode;
        this.spacingBottom = opts.spacingBottom;
        this.spacingLeft = opts.spacingLeft;
        this.spacingRight = opts.spacingRight;
        this.spacingTop = opts.spacingTop;
        this.hAlignSelf = opts.hAlignSelf;
        this.vAlignSelf = opts.vAlignSelf;
        this.flexGrow = opts.flexGrow;
        this.flexParams = opts.flexParams;
        this.isShowCaption = opts.isShowCaption;
        this.sysCss = opts.sysCss;
        this.predefinedType = opts.predefinedType;
        this.itemStyle = opts.itemStyle;
        this.sysImage = opts.sysImage;
        this.index = opts.index ? opts.index : 0;
        this.parentName = opts.parentName;
    }

    /**
     * 设置数据
     *
     * @param {*} value
     * @memberof PanelDetailModel
     */
    public setData(value: any) {
        this.data = value;
    }

    /**
     * 获取数据
     *
     * @return {*} 
     * @memberof PanelDetailModel
     */
    public getData() {
        return this.data;
    }

    /**
     * 设置当前下标
     *
     * @param {number} index
     * @memberof PanelDetailModel
     */
    public setIndex(index: number) {
        this.index = index;
    }

    /**
     * 获取当前下标
     *
     * @return {*} 
     * @memberof PanelDetailModel
     */
    public getIndex() {
        return this.index;
    }

    /**
     * 获取元素样式（除容器之外的元素，包含内容盒子 大小/边距/内容 的样式）
     *
     * @memberof PanelDetailModel
     */
    public getElementStyle() {
        const elementStyle = {};
        Object.assign(elementStyle, this.getBoxSizeStyle());
        Object.assign(elementStyle, this.getBoxSpacingStyle());
        Object.assign(elementStyle, this.getBoxSelfAlignStyle());
        return elementStyle;
    }

    /**
     * 获取盒子间隔样式（元素）
     *
     * @memberof PanelDetailModel
     */
    protected getBoxSpacingStyle() {
        const boxStyle = {};
        // 上方间隔模式
        if (this.spacingTop) {
            Object.assign(boxStyle, Util.getItemSpacingStyle(this.spacingTop, 'top'));
        }
        // 下方间隔模式
        if (this.spacingBottom) {
            Object.assign(boxStyle, Util.getItemSpacingStyle(this.spacingBottom, 'bottom'));
        }
        // 左侧间隔模式
        if (this.spacingLeft) {
            Object.assign(boxStyle, Util.getItemSpacingStyle(this.spacingLeft, 'left'));
        }
        // 右侧间隔模式
        if (this.spacingRight) {
            Object.assign(boxStyle, Util.getItemSpacingStyle(this.spacingRight, 'right'));
        }
        return boxStyle;
    }

    /**
     * 识别flex布局方向，横轴对齐，纵轴对齐（元素）
     *
     * @return {*} 
     * @memberof PanelDetailModel
     */
    protected getFlexStyle() {
        const boxStyle = {};
        if (this.flexParams.align || this.flexParams.dir || this.flexParams.vAlign) {
            Object.assign(boxStyle, { 'display': 'flex' });
            if (this.flexParams.dir) {
                Object.assign(boxStyle, { 'flex-direction': this.flexParams.dir });
            }
            if (this.flexParams.align) {
                Object.assign(boxStyle, { 'justify-content': this.flexParams.align });
            }
            if (this.flexParams.vAlign) {
                Object.assign(boxStyle, { 'align-items': this.flexParams.vAlign });
            }
        }
        return boxStyle;
    }

    /**
     * 获取盒子大小样式（元素）
     *
     * @memberof PanelDetailModel
     */
    protected getBoxSizeStyle() {
        const boxStyle = {};
        if (this.widthMode || this.layoutWidth) {
            Object.assign(boxStyle, Util.getBoxSize("WIDTH", this.widthMode, this.layoutWidth));
        }
        if (this.heightMode || this.layoutHeight) {
            Object.assign(boxStyle, Util.getBoxSize("HEIGHT", this.heightMode, this.layoutHeight));
        }
        return boxStyle;
    }

    /**
     * 获取自身对齐模式(元素)
     *
     * @memberof PanelDetailModel
     */
    protected getBoxSelfAlignStyle() {
        const boxStyle = {};
        // 自身对齐方式
        if (this.vAlignSelf || this.hAlignSelf) {
            // 自身垂直对齐模式
            switch (this.vAlignSelf) {
                case 'TOP':
                    // 默认样式
                    break;
                case 'MIDDLE':
                    Object.assign(boxStyle, { 'margin': 'auto 0px' });
                    break;
                case 'BOTTOM':
                    Object.assign(boxStyle, { 'margin-top': 'auto' });
                    break;
                default:
                    break;
            }
            // 自身水平对齐模式
            switch (this.hAlignSelf) {
                case 'LEFT':
                    Object.assign(boxStyle, { 'align-self': 'flex-start' });
                    break;
                case 'CENTER':
                    Object.assign(boxStyle, { 'align-self': 'center' });
                    break;
                case 'RIGHT':
                    Object.assign(boxStyle, { 'align-self': 'flex-end' });
                    break;
                case 'JUSTIFY':
                    // 默认样式
                    break;
                default:
                    break;
            }
        }
        return boxStyle;
    }

    /**
     * 获取盒子布局样式(布局,包含约束内容区布局的样式)
     *
     * @memberof PanelDetailModel
     */
    protected getBoxLayOutStyle() {
        const boxStyle = {};
        if (this.parentName && this.panel) {
            const parentItem = this.panel.layoutModelDetails[this.parentName];
            // 识别父FLEX
            if (parentItem) {
                if (parentItem.layout == 'FLEX') {
                    Object.assign(boxStyle, { 'display': 'contents' });
                    Object.assign(boxStyle, { 'flex-grow': this.flexGrow ? this.flexGrow : 0 });
                }
                // 识别SIMPLEFLEX
                else if (parentItem.layout == 'SIMPLEFLEX') {
                    // SIMPLEFLEX 按照内容撑大小
                    Object.assign(boxStyle, { 'flex-grow': 0 });
                    if (this.flexGrow !== -1) {
                        Object.assign(boxStyle, { 'width': `${(100 / 12) * this.flexGrow}%`, 'height': '100%' });
                    } else {
                        // 简单FLEX布局自适应
                        Object.assign(boxStyle, { 'flex-grow': 1, 'min-width': `${(100 / 12)}%`, 'height': '100%' });
                    }
                } else {
                    // 当前项兄弟项个数大于1且本身设置宽或者高默认不占满
                    if (parentItem && parentItem.details && parentItem.details.length > 1) {
                        const boxSizeStyle: any = this.getBoxSizeStyle();
                        if (boxSizeStyle && Object.keys(boxSizeStyle).length > 0) {
                            Object.assign(boxStyle, { 'flex-grow': 0 });
                        }
                    }
                }
            }
        }
        if (!this.visible) {
            Object.assign(boxStyle, { display: 'none' });
        }
        return boxStyle;
    }

    /**
     * 加载
     * @param context 
     * @param viewParams 
     */
    public async load(context: any, viewParams: any) { }

}